﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions; 
using System.Linq.Dynamic;
using System.Data.Linq; 
using System.Text;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.Utilities;

namespace VeteransAffairs.Registries.BusinessManager
{
    public class ContactLogManager:BaseBO
    {

        public ContactLogManager() 
        {
            _defaultSortField = "ReferralId desc";
            
        }

        private IQueryable<CONTACT_LOG> LinqAll()
        {
            //populate LinqAll
            IQueryable<CONTACT_LOG> tempLinq = (from e in _db.CONTACT_LOGs 
                                                join r in _db.REFERRALs.OfType<EFR_REFERRAL>() on e.REFERRAL_ID equals r.REFERRAL_ID   
                                                      select e);

            //TODO - add all business filtering rules 

            return tempLinq;

        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<CONTACT_LOG>(e => e.REFERRAL);
            lo.LoadWith<REFERRAL>(e => e.STD_REFERRALST);
            lo.LoadWith<REFERRAL>(e => e.PATIENT); 
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
       
        }

        public IEnumerable<Contact> SelectAll(string searchValue, string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                //IEnumerable<Contact> entities;
                var entities = SelectAllLinqFilter(searchValue).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }
        }


        public int SelectAllCount(string searchValue, string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                return SelectAllLinqFilter(searchValue).Count();
            }
        }

        private IQueryable<Contact> SelectAllLinqFilter(string value)
        {
            IQueryable<CONTACT_LOG> linqFilter = LinqAll();

            string fieldSearch = "LAST_NAME";

            var param = Expression.Parameter(typeof(CONTACT_LOG), "PATIENT");

            // e => e.REFERRAL.PATIENT.LAST_NAME.Contains()
            var name = Expression.PropertyOrField(Expression.PropertyOrField(Expression.PropertyOrField(param, "REFERRAL"), "PATIENT"), fieldSearch);
            var search = Expression.Constant(value, typeof(string));
            
            var body = Expression.Call(name, "Contains", null, search);

            Expression<Func<CONTACT_LOG, bool>> lambda;
            if (String.IsNullOrEmpty(value))
            {
                lambda = x => true;
            }
            else
            {
                lambda = Expression.Lambda<Func<CONTACT_LOG, bool>>(body, param);
            }


            return linqFilter.Where(lambda)

                .GroupBy(cl => new
                {
                    cl.REFERRAL_ID,
                    cl.REFERRAL.STD_REFERRALST.NAME,
                    cl.REFERRAL.REFERRAL_DATE,
                    cl.REFERRAL.PATIENT.LAST_NAME,
                    cl.REFERRAL.PATIENT.FIRST_NAME,
                    cl.REFERRAL.PATIENT.MIDDLE_NAME,
                    VAMCName = cl.REFERRAL.STD_INSTITUTION.NAME,
                    StationNumber = cl.REFERRAL.STD_INSTITUTION.STATIONNUMBER
                })
                .Select(grp => new Contact
                    {
                        ReferralId = grp.Key.REFERRAL_ID,
                        LastContactDate = grp.OrderByDescending(x => x.CONTACT_DATE).First().CONTACT_DATE.Value,
                        NumberOfContacts = grp.Count(),
                        ReferralStatus = grp.Key.NAME,
                        ReferralDate = grp.Key.REFERRAL_DATE.Value,
                        PatientName = string.IsNullOrEmpty(grp.Key.MIDDLE_NAME) ?
                            string.IsNullOrEmpty(grp.Key.FIRST_NAME) ? 
                                grp.Key.LAST_NAME : 
                                grp.Key.LAST_NAME + ", " + grp.Key.FIRST_NAME :
                            grp.Key.LAST_NAME + ", " + grp.Key.FIRST_NAME + " " + grp.Key.MIDDLE_NAME,
                        VAMC = grp.Key.VAMCName,
                        StationNumber = grp.Key.StationNumber,
                        PatientNameSort = grp.Key.LAST_NAME + " " + grp.Key.FIRST_NAME
                });
        }


        public IEnumerable<ContactLog> SelectByID(int id)
        {
             
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                var contactLog = SelectByIDLinqFilter(id).OrderByDescending(c => c.CONTACT_DATE).Select
                    (e => new ContactLog
                    {
                        ContactId = e.CONTACT_LOG_ID,
                        ContactName = e.CONTACT_LAST_NAME + ", " + e.CONTACT_FIRST_NAME,
                        ContactDate = String.Format("{0:g}", e.CONTACT_DATE.Value),
                        ContactReason = e.CONTACT_REASON_TEXT, 
                        ContactText = e.CONTACT_MESSAGE_TEXT,
                        ContactInitiator = e.CREATEDBY,
                        TelemedicinePatient = e.TELEMEDICINE_PATIENT_FLAG ,
                        TelemedicineProvider = e.TELEMEDICINE_PROVIDER_FLAG,
                        InpatientEvaluation = e.INPATIENT_EVAL_RECOM_FLAG,
                        Followup = e.FOLLOWUP_REQUIRED_FLAG
                        
                    }).ToList();

                return contactLog ;

            }

        }


        private IQueryable<CONTACT_LOG> SelectByIDLinqFilter(int id)
        {
            IQueryable<CONTACT_LOG> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.REFERRAL_ID  == id select t;

            return linqFilter;
        }


        public CONTACT_LOG SelectByContactID(int id)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                CONTACT_LOG contact = SelectByContactIDLinqFilter(id).SingleOrDefault();

                if (contact != null)
                {
                    contact.SetAsChangeTrackingRoot(true);

                    
                }
                return contact;
            }

        }

        private IQueryable<CONTACT_LOG> SelectByContactIDLinqFilter(int id)
        {
            IQueryable<CONTACT_LOG> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.CONTACT_LOG_ID == id select t;

            return linqFilter;
        }

        public int Update(CONTACT_LOG contact)
        {

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;

                contact.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.
                BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                    eventArgs.SavedItemId = contact.CONTACT_LOG_ID;
                    eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    eventArgs.SavedItemId = contact.CONTACT_LOG_ID;
                    eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                }
                catch (Exception e)
                {
                    eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                }
                RaiseSaveEvent(this, eventArgs);
            }
            return contact.CONTACT_LOG_ID;
        }

        public class ContactLog
        {
            public int ContactId { get; set; }
            public string ContactName { get; set; }
            public string ContactInitiator { get; set; }
            public string ContactDate { get; set; }
            public string ContactReason { get; set; }
            public string ContactText { get; set; }
            public bool TelemedicinePatient { get; set; }
            public bool TelemedicineProvider { get; set; }
            public bool InpatientEvaluation { get; set; }
            public bool Followup { get; set; }
            public string ContactComments
            {
                get { return String.Join(", ", new string[] { TelemedicinePatient ? "Telemedicine consultation with patient recommended" : String.Empty,
                           TelemedicineProvider ? "Telemedicine consultation with provider recommended" : String.Empty, Followup ? "Follow-up needed" : String.Empty,
                            InpatientEvaluation ? "Inpatient Evaluation Recommended" : String.Empty }.Where(s => !String.IsNullOrEmpty(s)).ToArray());
                }
            }    
        }
        public class Contact
        {
            public int ReferralId { get; set; }
            public string PatientName { get; set; }
            public DateTime ReferralDate { get; set; }
            public string ReferralStatus { get; set; }
            public int NumberOfContacts { get; set; }
            public DateTime LastContactDate { get; set; }
            public string VAMC { get; set; }
            public string StationNumber { get; set; }
            public string PatientNameSort { get; set; }
        }

    }
}
